import os
import csv
import json
import uuid
import requests
from build_payload import BuildPayload, FileUtils

#### CHECKLIST ####

# Define the webhook where you want to receive responses
WEBHOOK = "https://your.callback.com/here"

# You need to define the following environment variables in your terminal (export TEST_USERNAME=1234)
BASIC_AUTH_USERNAME = os.environ["TEST_USERNAME"]
BASIC_AUTH_PASSWORD = os.environ["TEST_PASSWORD"]

# You can dry run this script without sending payloads to Informed. If set to True, this will 
# print the payload in your terminal but will not send the payload or the images to Informed.
DRY_RUN = False

# All verifications barring the consumer verifications (consumer verifications are requested in the csv)
VERIFICATIONS_LIST = [
    "credit_application",
    "odometer_statement",
    "bookout_sheet",
    "gap_waiver_contract",
    "vehicle_service_contract"
]

if __name__ == "__main__":
    # must use python >= 3.9

    with open('../structured_data.json') as econ:
        structured_data = json.load(econ)

    # Informed endpoint (do not edit)
    base_url = "https://api.staging.informediq-infra.com/v1/auto/applications"
    
    # Pull data from csv file one row at a time
    with open("../test_cases.csv", "r") as appslist:
        reader = csv.DictReader(appslist)

        for row_count, row in enumerate(reader, start=1):
            # Build the full json structure with dictionary items for all elements
            app = BuildPayload.build(row, verifications=VERIFICATIONS_LIST, 
                                     print_unformatted_payload=False)
            
            app_ref_id = app["application_reference_id"]
            
            # Add the webhook defined above to the payload 
            app["webhook"] = WEBHOOK
            
            # Grab the file names associated with this app
            file_names = []
            try:
                file_names = FileUtils.get_file_names(prefix=f"../test_images/{app_ref_id}/")
            except:
                print(f"no file found in destination: ../test_images/{app_ref_id}")
            
            print("ALL FILES IN DIRECTORY: ", file_names)

            # Grab only image files
            img_files = []
            for f in file_names:
                tempfn = f.lower()
                if tempfn.endswith(".pdf") or tempfn.endswith(".tiff") or tempfn.endswith(".tif") \
                        or tempfn.endswith(".jpg") or tempfn.endswith(".jpeg") or tempfn.endswith(".png"):
                    img_files.append(f)

            # Update teh app ref id with a uuid to make it unique
            new_app_ref_id = f"Phase4_{app_ref_id}_{str(uuid.uuid4())}"
            app["application_reference_id"] = new_app_ref_id
            
            print(f"row {row_count}: {new_app_ref_id}")
            print("\nALL ACCEPTABLE IMAGE FILES: ", img_files)

            if not img_files:
                raise Exception(f"No images found for app_ref_id:  {app_ref_id} **")                
            
            print("\n\nFORMATTED PAYLOAD:")
            print(json.dumps(app, indent=2))  # print the final payload
            print("app_ref_id: ", new_app_ref_id)
            print("\n------------------\n")
                
            # STEP #1: POST payload to the endpoint to create the app
            if not DRY_RUN:  
                print(f"Sending create app requests to {base_url}\n")
                api_call = requests.post(url=base_url,
                                         auth=(BASIC_AUTH_USERNAME,
                                               BASIC_AUTH_PASSWORD),
                                         json=app)

                print(f"\napi_call status:  {api_call.status_code}")
                api_call.raise_for_status()
                # Grab application id from the synchronous response or deal with the error message  
                initial_resp = api_call.json()
                informed_app_id = initial_resp["application_id"]
                print("informed_application_id: ", informed_app_id)
                print(f"received synchronous response for app creation:\n{json.dumps(initial_resp, indent=2)}\n")                   
                print(f"app_ref_id:  {new_app_ref_id}, informed_application_id:  {informed_app_id}")
            else:
                print("Application not sent to Informed")

            # STEP #2: send list of image files to Informed 
            if not DRY_RUN:
                docs_req_endpoint = f"{base_url}/{informed_app_id}/documents"
                docs_payload = {
                    "image_files": img_files,
                    "structured_data": structured_data
                }

                print(json.dumps(docs_payload, indent=2))
                print(f"Sending file list and structured data to {docs_req_endpoint}\n")

                # POST the list of files and structured_data to the create_documents endpoint
                docs_req = requests.post(url=docs_req_endpoint,
                                         auth=(BASIC_AUTH_USERNAME,
                                               BASIC_AUTH_PASSWORD),
                                         json=docs_payload)

                print(f"\napi_call status:  {docs_req.status_code}")
                docs_req.raise_for_status()
                docs_resp = docs_req.json()
                print(f"received new synchronous response for image files:\n{json.dumps(docs_resp, indent=2)}\n")
            else:
                print("File list not sent to Informed")

            # STEP #3: Upload the image files to the URLs received in step #2 above
            if not DRY_RUN:
                # Receive pre-signed URLs in the synchronous response and then push files to associated URLs                            
                for resrc in docs_resp["image_files"]:                        
                    if resrc["file_reference_id"] in docs_payload["image_files"]:
                        print(f"pushing\n {resrc['file_reference_id']} \nto\n {resrc['url']}")

                        local_file = resrc["file_reference_id"]
                        print(f"as\n{local_file}\n")

                        with open(local_file, 'rb') as data:
                            r = requests.put(url=resrc["url"], data=data)

                            print(f"push status:  {r.status_code}\n")
                            r.raise_for_status()

            else:
                print("Files not sent to Informed")

            print("\n------------------\n")
